﻿Option Strict On
Option Explicit On

Imports System.Drawing
Imports System.Windows.Forms

Public Class Form1
    Inherits Form

    ' --- Game state ---
    Private gameTimer As Timer
    Private rand As New Random()
    Private player2 As Player
    Private bullets As New List(Of Bullet)()
    Private enemies As New List(Of Enemy)()
    Private explosions As New List(Of Explosion)()
    Private score As Integer = 0
    Private level As Integer = 1
    Private spawnCounter As Integer = 0
    Private isRunning As Boolean = False
    Private isPaused As Boolean = False

    ' Controls
    Private lblScore As Label
    Private lblLives As Label
    Private lblLevel As Label

    Public Sub New()
        Me.Text = "Aerial Warfare - VB.NET"
        Me.ClientSize = New Size(800, 600)
        Me.DoubleBuffered = True
        Me.FormBorderStyle = FormBorderStyle.FixedSingle
        Me.MaximizeBox = False
        Me.StartPosition = FormStartPosition.CenterScreen
        Me.CenterToScreen()

        InitializeUI()
        InitializeGame()

        ' Timer: main game loop (60 FPS)
        gameTimer = New Timer()
        gameTimer.Interval = 16 ' ~60 FPS
        AddHandler gameTimer.Tick, AddressOf GameLoop

        ' Key events
        AddHandler Me.KeyDown, AddressOf OnKeyDown
        AddHandler Me.KeyUp, AddressOf OnKeyUp
    End Sub

    Private Sub InitializeUI()
        lblScore = New Label() With {
            .AutoSize = True,
            .Font = New Font("Segoe UI", 10, FontStyle.Bold),
            .Location = New Point(10, 10)
        }
        Me.Controls.Add(lblScore)

        lblLives = New Label() With {
            .AutoSize = True,
            .Font = New Font("Segoe UI", 10, FontStyle.Bold),
            .Location = New Point(10, 30)
        }
        Me.Controls.Add(lblLives)

        lblLevel = New Label() With {
            .AutoSize = True,
            .Font = New Font("Segoe UI", 10, FontStyle.Bold),
            .Location = New Point(10, 50)
        }
        Me.Controls.Add(lblLevel)

    End Sub

    Private Sub InitializeGame()
        player2 = New Player(New PointF(CSng(Me.ClientSize.Width / 2), Me.ClientSize.Height - 80))
        bullets.Clear()
        enemies.Clear()
        explosions.Clear()
        score = 0
        level = 1
        spawnCounter = 0
        isRunning = False
        isPaused = False
        UpdateUILabels()
    End Sub

    Private Sub StartGame()
        InitializeGame()
        isRunning = True
        gameTimer.Start()
        Me.Focus()
    End Sub

    Private Sub RestartGame()
        StartGame()
    End Sub

    Private Sub TogglePause()
        If Not isRunning Then Return
        isPaused = Not isPaused
    End Sub

    Private Sub EndGame()
        isRunning = False
        gameTimer.Stop()
        Invalidate()
    End Sub

    Private Sub GameLoop(sender As Object, e As EventArgs)
        If Not isRunning OrElse isPaused Then
            Invalidate()
            Return
        End If

        ' Update player (movement handled by key events)
        player2.Update(Me.ClientSize)

        ' Spawn enemies based on level
        spawnCounter += 1
        Dim spawnRate As Integer = Math.Max(40 - (level - 1) * 3, 12)
        If spawnCounter >= spawnRate Then
            spawnCounter = 0
            SpawnEnemy()
        End If

        ' Update bullets
        For i As Integer = bullets.Count - 1 To 0 Step -1
            bullets(i).Update()
            If bullets(i).Position.Y < -50 Then
                bullets.RemoveAt(i)
            End If
        Next

        ' Update enemies
        For i As Integer = enemies.Count - 1 To 0 Step -1
            enemies(i).Update()
            ' Remove if off-screen
            If enemies(i).Position.Y > Me.ClientSize.Height + 50 OrElse enemies(i).Position.X < -100 OrElse enemies(i).Position.X > Me.ClientSize.Width + 100 Then
                enemies.RemoveAt(i)
                Continue For
            End If
        Next

        ' Update explosions
        For i As Integer = explosions.Count - 1 To 0 Step -1
            explosions(i).Update()
            If explosions(i).IsFinished Then explosions.RemoveAt(i)
        Next

        ' Collisions: bullets vs enemies
        For b As Integer = bullets.Count - 1 To 0 Step -1
            Dim bulletRect = bullets(b).Bounds
            Dim removedBullet As Boolean = False
            For en As Integer = enemies.Count - 1 To 0 Step -1
                If bulletRect.IntersectsWith(enemies(en).Bounds) Then
                    enemies(en).Health -= bullets(b).Damage
                    bullets.RemoveAt(b)
                    removedBullet = True
                    If enemies(en).Health <= 0 Then
                        score += enemies(en).ScoreValue
                        explosions.Add(New Explosion(enemies(en).Position))
                        enemies.RemoveAt(en)
                        ' Level up simple logic
                        If score >= level * 100 Then
                            level += 1
                        End If
                    End If
                    Exit For
                End If
            Next
            If removedBullet Then Continue For
        Next

        ' Collisions: player vs enemies
        For en As Integer = enemies.Count - 1 To 0 Step -1
            If player2.Bounds.IntersectsWith(enemies(en).Bounds) Then
                ' hurt player
                player2.Lives -= 1
                explosions.Add(New Explosion(player2.Position))
                enemies.RemoveAt(en)
                If player2.Lives <= 0 Then
                    EndGame()
                End If
            End If
        Next

        UpdateUILabels()
        Invalidate() ' trigger Paint
    End Sub

    Private Sub SpawnEnemy()
        Dim x As Integer = rand.Next(30, Me.ClientSize.Width - 30)
        Dim typeChance = rand.NextDouble()
        Dim enemy As Enemy
        If typeChance < 0.6 Then
            enemy = New Enemy(New PointF(x, -40), EnemyType.Light, level)
        ElseIf typeChance < 0.9 Then
            enemy = New Enemy(New PointF(x, -60), EnemyType.Medium, level)
        Else
            enemy = New Enemy(New PointF(x, -80), EnemyType.Heavy, level)
        End If
        enemies.Add(enemy)
    End Sub

    Private Sub UpdateUILabels()
        lblScore.Text = $"Score: {score}"
        lblLives.Text = $"Lives: {player2.Lives}"
        lblLevel.Text = $"Level: {level}"
    End Sub

    ' --- Input management ---
    Private keysPressed As New HashSet(Of Keys)()

    Private Sub OnKeyDown(sender As Object, e As KeyEventArgs)
        If Not isRunning Then
            If e.KeyCode = Keys.Enter Then StartGame()
            Return
        End If

        keysPressed.Add(e.KeyCode)
        If e.KeyCode = Keys.Space Then
            ' Fire
            Dim b As Bullet = player2.Shoot()
            If b IsNot Nothing Then bullets.Add(b)
        ElseIf e.KeyCode = Keys.P Then
            TogglePause()
        End If

        UpdateMovementFromKeys()
    End Sub

    Private Sub OnKeyUp(sender As Object, e As KeyEventArgs)
        If keysPressed.Contains(e.KeyCode) Then keysPressed.Remove(e.KeyCode)
        UpdateMovementFromKeys()
    End Sub

    Private Sub UpdateMovementFromKeys()
        Dim vx As Single = 0
        Dim vy As Single = 0
        If keysPressed.Contains(Keys.Left) OrElse keysPressed.Contains(Keys.A) Then vx -= 1
        If keysPressed.Contains(Keys.Right) OrElse keysPressed.Contains(Keys.D) Then vx += 1
        If keysPressed.Contains(Keys.Up) OrElse keysPressed.Contains(Keys.W) Then vy -= 1
        If keysPressed.Contains(Keys.Down) OrElse keysPressed.Contains(Keys.S) Then vy += 1
        player2.SetInputVelocity(vx, vy)
    End Sub

    ' --- Rendering ---
    Protected Overrides Sub OnPaint(e As PaintEventArgs)
        Dim g As Graphics = e.Graphics
        g.SmoothingMode = Drawing2D.SmoothingMode.AntiAlias

        ' Background: simple gradient sky
        Using b As New Drawing2D.LinearGradientBrush(Me.ClientRectangle, Color.LightSkyBlue, Color.CornflowerBlue, 90.0F)
            g.FillRectangle(b, Me.ClientRectangle)
        End Using

        ' Draw player
        If isRunning OrElse player2.Lives > 0 Then player2.Draw(g)

        ' Draw bullets
        For Each b In bullets
            b.Draw(g)
        Next

        ' Draw enemies
        For Each en In enemies
            en.Draw(g)
        Next

        ' Draw explosions
        For Each ex In explosions
            ex.Draw(g)
        Next

        ' Overlay: paused or game over
        If Not isRunning AndAlso player2.Lives <= 0 Then
            DrawCenteredText(g, "GAME OVER", New Font("Segoe UI", 36, FontStyle.Bold), Brushes.White, New PointF(CSng(Me.ClientSize.Width / 2), CSng(Me.ClientSize.Height / 2 - 40)))
            DrawCenteredText(g, $"Final Score: {score}", New Font("Segoe UI", 18, FontStyle.Bold), Brushes.White, New PointF(CSng(Me.ClientSize.Width / 2), CSng(Me.ClientSize.Height / 2 + 10)))
            DrawCenteredText(g, "Press Start to play again", New Font("Segoe UI", 12), Brushes.White, New PointF(CSng(Me.ClientSize.Width / 2), CSng(Me.ClientSize.Height / 2 + 50)))
        ElseIf isPaused Then
            DrawCenteredText(g, "PAUSED", New Font("Segoe UI", 36, FontStyle.Bold), Brushes.White, New PointF(CSng(Me.ClientSize.Width / 2), CSng(Me.ClientSize.Height / 2 - 20)))
        ElseIf Not isRunning Then
            DrawCenteredText(g, "Aerial Warfare", New Font("Segoe UI", 28, FontStyle.Bold), Brushes.White, New PointF(CSng(Me.ClientSize.Width / 2), CSng(Me.ClientSize.Height / 2 - 40)))
            DrawCenteredText(g, "Press Start (or Enter) to begin", New Font("Segoe UI", 14), Brushes.White, New PointF(CSng(Me.ClientSize.Width / 2), CSng(Me.ClientSize.Height / 2 + 10)))
            DrawCenteredText(g, "Move: Arrow keys or WASD • Shoot: Space • Pause: P", New Font("Segoe UI", 10), Brushes.White, New PointF(CSng(Me.ClientSize.Width / 2), CSng(Me.ClientSize.Height / 2 + 40)))
        End If

        MyBase.OnPaint(e)
    End Sub

    Private Sub DrawCenteredText(g As Graphics, text As String, font As Font, brush As Brush, center As PointF)
        Dim size = g.MeasureString(text, font)
        g.DrawString(text, font, Brushes.Black, center.X - size.Width / 2 + 2, center.Y - size.Height / 2 + 2)
        g.DrawString(text, font, brush, center.X - size.Width / 2, center.Y - size.Height / 2)
    End Sub

    ' --- Entity classes ---
    Private Class Player
        Public Property Position As PointF
        Public Property Speed As Single = 5.5F
        Public Property Lives As Integer = 5
        Private inputVX As Single = 0
        Private inputVY As Single = 0
        Private fireCooldown As Integer = 0

        Public Sub New(pos As PointF)
            Me.Position = pos
        End Sub

        Public ReadOnly Property Bounds As RectangleF
            Get
                Return New RectangleF(Position.X - 18, Position.Y - 18, 36, 36)
            End Get
        End Property

        Public Sub SetInputVelocity(vx As Single, vy As Single)
            Me.inputVX = vx
            Me.inputVY = vy
        End Sub

        Public Sub Update(clientSize As Size)
            ' Normalize diagonal speed
            Dim vx = inputVX
            Dim vy = inputVY
            If vx <> 0 Or vy <> 0 Then
                Dim len = CSng(Math.Sqrt(vx * vx + vy * vy))
                vx = vx / len
                vy = vy / len
            End If
            Position = New PointF(Position.X + vx * Speed, Position.Y + vy * Speed)

            ' Clamp to client bounds
            Position = New PointF(Math.Max(20, Math.Min(clientSize.Width - 20, Position.X)),
                                  Math.Max(20, Math.Min(clientSize.Height - 20, Position.Y)))

            If fireCooldown > 0 Then fireCooldown -= 1
        End Sub

        Public Function Shoot() As Bullet
            If fireCooldown > 0 Then Return Nothing
            fireCooldown = 10 ' cooldown frames
            ' create two bullets slightly offset
            Return New Bullet(New PointF(Position.X, Position.Y - 24), New PointF(0, -10), 8)
        End Function

        Public Sub Draw(g As Graphics)
            ' Draw simple plane as triangle
            Dim p1 = New PointF(Position.X, Position.Y - 16)
            Dim p2 = New PointF(Position.X - 16, Position.Y + 12)
            Dim p3 = New PointF(Position.X + 16, Position.Y + 12)
            Dim pts() As PointF = {p1, p2, p3}
            Using brush As New SolidBrush(Color.Yellow)
                g.FillPolygon(brush, pts)
            End Using
            ' cockpit
            Using pen As New Pen(Color.Black, 1.2F)
                g.DrawPolygon(pen, pts)
                g.DrawEllipse(pen, Position.X - 4, Position.Y - 2, 8, 6)
            End Using
        End Sub
    End Class

    Private Class Bullet
        Public Property Position As PointF
        Public Property Velocity As PointF
        Public Property Damage As Integer

        Public Sub New(pos As PointF, vel As PointF, dmg As Integer)
            Me.Position = pos
            Me.Velocity = vel
            Me.Damage = dmg
        End Sub

        Public Sub Update()
            Position = New PointF(Position.X + Velocity.X, Position.Y + Velocity.Y)
        End Sub

        Public ReadOnly Property Bounds As RectangleF
            Get
                Return New RectangleF(Position.X - 4, Position.Y - 8, 8, 16)
            End Get
        End Property

        Public Sub Draw(g As Graphics)
            g.FillRectangle(Brushes.White, Position.X - 2, Position.Y - 8, 4, 12)
        End Sub
    End Class

    Private Enum EnemyType
        Light
        Medium
        Heavy
    End Enum

    Private Class Enemy
        Public Property Position As PointF
        Public Property Velocity As PointF
        Public Property Health As Integer
        Public Property Type As EnemyType
        Public Property ScoreValue As Integer

        Public Sub New(pos As PointF, type As EnemyType, level As Integer)
            Me.Position = pos
            Me.Type = type
            Dim baseSpeed As Single = 2.0F
            Select Case type
                Case EnemyType.Light
                    Health = 8
                    ScoreValue = 10
                    Velocity = New PointF(0, baseSpeed + 0.6F + level * 0.05F)
                Case EnemyType.Medium
                    Health = 18
                    ScoreValue = 25
                    Velocity = New PointF(CSng((New Random()).NextDouble() - 0.5) * 2.5F, baseSpeed + 0.4F + level * 0.04F)
                Case EnemyType.Heavy
                    Health = 40
                    ScoreValue = 60
                    Velocity = New PointF(0, baseSpeed + 0.2F + level * 0.03F)
            End Select
        End Sub

        Public Sub Update()
            Position = New PointF(Position.X + Velocity.X, Position.Y + Velocity.Y)
            ' slight horizontal wiggle for medium type
            If Type = EnemyType.Medium Then
                Position = New PointF(Position.X + CSng(Math.Sin(Position.Y / 12.0F)), Position.Y)
            End If
        End Sub

        Public ReadOnly Property Bounds As RectangleF
            Get
                Dim size As Integer = If(Type = EnemyType.Light, 30, If(Type = EnemyType.Medium, 42, 64))
                Return New RectangleF(CSng(Position.X - size / 2), CSng(Position.Y - size / 2), size, size)
            End Get
        End Property

        Public Sub Draw(g As Graphics)
            Dim rect = Bounds
            Select Case Type
                Case EnemyType.Light
                    Using br As New SolidBrush(Color.OrangeRed)
                        g.FillEllipse(br, rect)
                    End Using
                Case EnemyType.Medium
                    Using br As New SolidBrush(Color.DarkRed)
                        g.FillEllipse(br, rect)
                    End Using
                Case EnemyType.Heavy
                    Using br As New SolidBrush(Color.Maroon)
                        g.FillEllipse(br, rect)
                    End Using
            End Select
            Using pen As New Pen(Color.Black, 1)
                g.DrawEllipse(pen, rect)
            End Using
        End Sub
    End Class

    Private Class Explosion
        Public Property Position As PointF
        Private life As Integer = 0
        Private maxLife As Integer = 18

        Public Sub New(pos As PointF)
            Me.Position = pos
            life = 0
        End Sub

        Public Sub Update()
            life += 1
        End Sub

        Public ReadOnly Property IsFinished As Boolean
            Get
                Return life >= maxLife
            End Get
        End Property

        Public Sub Draw(g As Graphics)
            Dim t = life / CSng(maxLife)
            Dim size = 8 + t * 48
            Dim alpha = CInt(255 * (1 - t))
            alpha = Math.Max(0, Math.Min(255, alpha))
            Dim c = Color.FromArgb(alpha, Color.Orange)
            Using br As New SolidBrush(c)
                g.FillEllipse(br, Position.X - size / 2, Position.Y - size / 2, size, size)
            End Using
        End Sub
    End Class

    ' --- Boilerplate: start the app ---
    <STAThread()>
    Public Shared Sub Main()
        Application.EnableVisualStyles()
        Application.SetCompatibleTextRenderingDefault(False)
        Application.Run(New Form1())
    End Sub

End Class

